<?php

namespace App\Services;

use App\Api\OpenCartClient;
use Exception;

class CategoryService {
    private $client;
    private $logManager;

    public function __construct() {
        $this->client = new OpenCartClient();
        $this->logManager = new LogManager(); // Initialize LogManager
    }

    /**
     * Process categories from multiple CSV files and sync with OpenCart
     */
    public function processCategories() {
        $supplier = 'supplier1'; // Define supplier name
        try {
            // Get the absolute path to the CSV files
            $categoryWebsiteCsv = __DIR__ . '/../../data/supplier1/Category-Website.csv';
            $categoryWebsiteExtrasCsv = __DIR__ . '/../../data/supplier1/Category-Website-Extras.csv';

            // Log the start of category processing
            $this->logManager->logInfo("Started processing categories", __FUNCTION__, $supplier);

            // Process main categories from Category-Website.csv
            $this->processParentCategories($categoryWebsiteCsv, $supplier);

            // Process ECO categories separately from Category-Website-Extras.csv
            $this->processEcoCategories($categoryWebsiteExtrasCsv, $supplier);

            // Log the completion of category processing
            $this->logManager->logInfo("Completed processing categories", __FUNCTION__, $supplier);
        } catch (Exception $e) {
            // Log any exceptions that occur during category processing
            $this->logManager->logError("Error processing categories: " . $e->getMessage(), __FUNCTION__, $supplier);
        }
    }

    /**
     * Process parent categories and subcategories from Category-Website.csv
     * @param string $csvFile
     * @param string $supplier
     */
    private function processParentCategories($csvFile, $supplier) {
        try {
            $categoriesFromCsv = $this->parseCsv($csvFile);
            $categoriesFromApi = $this->client->getCategoriesByLevel(2); // Fetch categories from OpenCart API

            foreach ($categoriesFromCsv as $parentCategory => $subCategories) {
                // Skip ECO parent category during parent category processing
                if (strcasecmp($parentCategory, 'ECO') === 0) {
                    continue;
                }

                // Log the start of processing for each parent category
                $this->logManager->logInfo("Processing parent category: $parentCategory", __FUNCTION__, $supplier);

                // Get parent category ID from the API
                $parentCategoryId = $this->client->getCategoryIdByName($parentCategory);

                foreach ($subCategories as $subCategory) {
                    // Check if subcategory exists in the API
                    $subcategoryExists = $this->checkIfCategoryExists($subCategory, $categoriesFromApi, $parentCategoryId);

                    // Prepare the data object
                    $categoryData = $this->prepareCategoryData($subCategory, $parentCategoryId);

                    if ($subcategoryExists) {
                        // Log the edit action for subcategory
                        $this->logManager->logInfo("Editing subcategory: $subCategory", __FUNCTION__, $supplier);
                        $this->client->editCategory($subcategoryExists['category_id'], $categoryData);
                    } else {
                        // Log the add action for subcategory
                        $this->logManager->logInfo("Adding new subcategory: $subCategory", __FUNCTION__, $supplier);
                        $this->client->addCategory($categoryData);
                    }
                }

                // Check for subcategories to delete, excluding ECO-related subcategories
                $this->removeDeletedSubcategories($parentCategoryId, $categoriesFromCsv[$parentCategory], $categoriesFromApi, 'ECO', $supplier);
            }
        } catch (Exception $e) {
            // Log any errors that occur during parent category processing
            $this->logManager->logError("Error processing parent categories: " . $e->getMessage(), __FUNCTION__, $supplier);
        }
    }

    /**
     * Process ECO categories from Category-Website-Extras.csv
     * @param string $csvFile
     * @param string $supplier
     */
    private function processEcoCategories($csvFile, $supplier) {
        try {
            $ecoCategoriesFromCsv = $this->parseCsv($csvFile);
            $ecoParentId = $this->client->getCategoryIdByName('ECO');

            foreach ($ecoCategoriesFromCsv['ECO'] as $ecoSubcategory) {
                // Check if ECO subcategory exists
                $ecoSubcategoryExists = $this->checkIfCategoryExists($ecoSubcategory, $this->client->getCategoriesByLevel(2), $ecoParentId);

                // Prepare the data object
                $categoryData = $this->prepareCategoryData($ecoSubcategory, $ecoParentId);

                if ($ecoSubcategoryExists) {
                    // Log the edit action for ECO subcategory
                    $this->logManager->logInfo("Editing ECO subcategory: $ecoSubcategory", __FUNCTION__, $supplier);
                    $this->client->editCategory($ecoSubcategoryExists['category_id'], $categoryData);
                } else {
                    // Log the add action for ECO subcategory
                    $this->logManager->logInfo("Adding new ECO subcategory: $ecoSubcategory", __FUNCTION__, $supplier);
                    $this->client->addCategory($categoryData);
                }
            }

            // Remove ECO-related subcategories only
            $this->removeDeletedSubcategories($ecoParentId, $ecoCategoriesFromCsv['ECO'], $this->client->getCategoriesByLevel(2), '', $supplier);
        } catch (Exception $e) {
            // Log any errors that occur during ECO category processing
            $this->logManager->logError("Error processing ECO categories: " . $e->getMessage(), __FUNCTION__, $supplier);
        }
    }

    /**
     * Remove subcategories from the website that don't exist in the CSV, excluding ECO-related subcategories
     * @param int $parentCategoryId
     * @param array $csvSubcategories
     * @param array $categoriesFromApi
     * @param string $excludeParent
     * @param string $supplier
     */
    private function removeDeletedSubcategories($parentCategoryId, $csvSubcategories, $categoriesFromApi, $excludeParent, $supplier) {
        try {
            foreach ($categoriesFromApi as $apiCategory) {
                if ($apiCategory['category_id'] === $parentCategoryId && !empty($apiCategory['categories'])) {
                    foreach ($apiCategory['categories'] as $subCategoryData) {
                        if (!in_array($subCategoryData['name'], $csvSubcategories)) {
                            // Only delete if the parent category is not excluded (e.g., not ECO)
                            if (strcasecmp($apiCategory['name'], $excludeParent) !== 0  && strcasecmp($apiCategory['column'], $excludeParent) !== 20) {
                                // Log the deletion of the subcategory
                                $this->logManager->logInfo("Deleting subcategory: " . $subCategoryData['name'], __FUNCTION__, $supplier);
                                $this->client->deleteCategory($subCategoryData['category_id']);
                            }
                        }
                    }
                }
            }
        } catch (Exception $e) {
            // Log any errors that occur during subcategory deletion
            $this->logManager->logError("Error removing deleted subcategories: " . $e->getMessage(), __FUNCTION__, $supplier);
        }
    }

    /**
     * Parse CSV and return categories structure
     * @param string $csvFile
     * @return array
     */
    private function parseCsv($csvFile) {
        $categories = [];
        $file = new \SplFileObject($csvFile);
        $file->setFlags(\SplFileObject::READ_CSV);
    
        $headers = [];
        foreach ($file as $index => $row) {
            if ($index === 0) {
                $headers = $row; // First row for parent categories
                continue;
            }
    
            foreach ($row as $colIndex => $subcategory) {
                if (!empty($subcategory)) {
                    $parentCategory = $headers[$colIndex];
                    $categories[$parentCategory][] = $subcategory;
                }
            }
        }
        return $categories;
    }
    
    // Non Optimized
    // private function parseCsv($csvFile) {
    //     $categories = [];
    //     try {
    //         if (($handle = fopen($csvFile, 'r')) !== false) {
    //             $headers = fgetcsv($handle); // First row for parent categories

    //             while (($row = fgetcsv($handle)) !== false) {
    //                 foreach ($row as $index => $subcategory) {
    //                     if (!empty($subcategory)) {
    //                         $parentCategory = $headers[$index];
    //                         $categories[$parentCategory][] = $subcategory;
    //                     }
    //                 }
    //             }
    //             fclose($handle);
    //         }
    //         $this->logManager->logInfo("Successfully parsed CSV file: $csvFile", __FUNCTION__, 'supplier1');
    //     } catch (Exception $e) {
    //         // Log any errors that occur during CSV parsing
    //         $this->logManager->logError("Error parsing CSV file: " . $e->getMessage(), __FUNCTION__, 'supplier1');
    //     }
    //     return $categories;
    // }

    /**
     * Check if the subcategory exists under the parent category in the OpenCart API
     * @param string $subcategory
     * @param array $categoriesFromApi
     * @param int $parentCategoryId
     * @return array|null
     */
    private function checkIfCategoryExists($subcategory, $categoriesFromApi, $parentCategoryId) {
        foreach ($categoriesFromApi as $apiCategory) {
            if ($apiCategory['category_id'] === $parentCategoryId && !empty($apiCategory['categories'])) {
                foreach ($apiCategory['categories'] as $subCategoryData) {
                    if (strcasecmp($subCategoryData['name'], $subcategory) === 0) {
                        return $subCategoryData;
                    }
                }
            }
        }
        return null;
    }

    /**
     * Prepare category data object for API
     * @param string $categoryName
     * @param int $parentCategoryId
     * @return array
     */
    private function prepareCategoryData($categoryName, $parentCategoryId) {
        return [
            "category_description" => [
                [
                    "name" => $categoryName,
                    "description" => "", // Empty description
                    "language_id" => 1,   // English language
                    "meta_description" => $categoryName,
                    "meta_keyword" => $categoryName,
                    "meta_title" => $categoryName
                ]
            ],
            "image" => "",
            "sort_order" => 0,
            "category_store" => [0],
            "parent_id" => $parentCategoryId,
            "status" => 1,
            "category_filter" => [0],
            "column" => 1,
            "top" => 0,
            "category_seo_url" => [
                [
                    "keyword" => '',
                    "language_id" => 1,
                    "store_id" => 0
                ]
            ]
        ];
    }
}
